//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Low-wealth outflows - HILDA
// Objective: 	Estimating the wealth at death of individuals who are not
//				expected to go through the probate system. These results will
//				supplement the results from administrative datasets to calculate
//				the administrative estimate of inheritance outflows.
//
// Created: 	13/08/2021
// Modified: 	15/11/2021
//==============================================================================

***************************************
*** Importing the Data ***
***************************************

clear

cd "XXXX"	// set the content directory here

use HILDA_restricted_combined_rescaled	// importing the combined dataset that contains all the different weights for nursing home residents

gen pw_net_worth = pwassei - pwdebti	// generating a personal net worth variable as the difference between personal assets and personal debts

gen ex_super = pw_net_worth - pwsupei	// generating a personal net worth variable that excludes superannuation assets
replace ex_super = 0 if ex_super <= 0	// replacing net worth excluding superannuation as zero if it was negative because negative wealth cannot be passed on

gen net_property = pwtpvi - pwtpdi		// generating a net property variable

gen married = 0							
replace married = 1 if mrcurr == 1		// generating an indicator variable for marital status

gen death_age = hgage						// generating a 'death_age' variable to allow matching with death rates
replace death_age = 100 if death_age > 100	// setting the maximum 'death_age' equal to 100 because that is the last age for which death rate data is available.

keep xwaveid wave hhpxid pwobani pwjbani pwsupri pwsupwi pwoccdi pwjccdi pwhecdi pwothdi pwhmvai pwhmdti pwopvai pwopdti pwcbani pwcaini pweqini pwbusvi pwbusdi pwtrusi pwinsui pwcolli pwvechi pwobdti pwtbani pwsupei pwtpvi pwccdti pwtpdi pwfini pwnfii pwassei pwdebti hgage hgsex benchmark_ratio benchmarked* pw_net_worth ex_super net_property death_age married	// keeping all relevant variables

destring xwaveid, replace
destring hhpxid, replace

**********************************
*** Merging in Death Rate Data ***
**********************************

gen year = wave + 2000	// generating a year variable to enable merging with the age and gender specific death rates

merge m:1 year death_age hgsex married using "Death_rates_by_marital_status.dta", nogen	// merging in the age and gender specific death rate data

*************************
*** Cleaning the Data ***
*************************

drop if (wave != 2 & wave != 6 & wave != 10 & wave != 14 & wave != 18)	//drops waves that don't have a wealth module

codebook pw_net_worth
replace pw_net_worth = 0 if pw_net_worth < 0	// replacing negative values of net worth with zero because negative wealth cannot be passed on

save HILDA_wealth_low_value, replace			// saving the dataset
save HILDA_partner_wealth_low_value, replace	// saving a duplicate of the dataset to be used later to match partner death rates

**************************************
*** Weighting the Wealth Variables ***
**************************************

local wealth_vars pwobani pwjbani pwsupri pwsupwi pwoccdi pwjccdi pwhecdi pwothdi pwhmvai pwhmdti pwopvai pwopdti pwcbani pwcaini pweqini pwbusvi pwbusdi pwtrusi pwinsui pwcolli pwvechi pwobdti pwtbani pwsupei pwtpvi pwccdti pwtpdi pwfini pwnfii pwassei pwdebti pw_net_worth ex_super	// creating a local macro that includes all personal wealth variables

foreach w of local wealth_vars {
	gen dw_bm_new_hhwte_`w' = `w' * benchmarked_new_hhwte * deathrate	// benchmark-weighted aggregate wealth
}	// multiplying each of the personal wealth variables by the age- and sex-specific death rates as well as the different sample and population weights

gen benchmarked_new_hhwte_deathrate = deathrate * benchmarked_new_hhwte	// multiplying the death rate variable by the benchmarked population-level enumerated person weights that include residents of non-private dwellings

************
keep if (net_property <= 0 & pwtbani <= 50000 & pweqini <= 50000 | ex_super <= 0)	// excluding those whose death is unlikely to result in a probate application
************

collapse(sum) `wealth_vars' dw* deathrate benchmarked_new_hhwte_deathrate, by(year)	// collapsing (summing) the weighted and unweighted wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

save HILDA_wealth_at_death_low_value, replace	// saving the big dataset

keep year dw_bm_new_hhwte_ex_super dw_bm_new_hhwte_pw_net_worth deathrate benchmarked_new_hhwte_deathrate	// keeping variables that will be used for plotting

save HILDA_estates_low_value, replace	// saving the small dataset

*****************************************
*** Importing the Partner Wealth Data ***
*****************************************

clear

use HILDA_partner_wealth_low_value	// opening the dataset

* For Australian resident comparison
// use HILDA_restricted_partner_wealth_domestic_children

keep year xwaveid hhpxid deathrate	// keeping relevant identifiers and the death rate variable

drop if hhpxid == .	// dropping observations for those that do not have partners
drop hhpxid	// dropping the partner identifier variable

rename xwaveid hhpxid	// renaming the individual identifier to be the partner identifier
rename deathrate partner_deathrate	// renaming the death rate variable to allow it to be merged with the main dataset

save HILDA_partner_deathrates_low_value, replace	// saving the dataset

************************************************
*** Merging the Partner Death Rate Variables ***
************************************************

clear

use HILDA_wealth_low_value	// opening the dataset

merge m:1 hhpxid year using HILDA_partner_deathrates_low_value	// merging in the partner death rates

replace deathrate = deathrate * partner_deathrate if partner_deathrate != .	// for couples, replacing the likelihood of death as the likelihood that both members of a couple die in a given year
rename deathrate final_deathrate

**************************************
*** Weighting the Wealth Variables ***
**************************************

local wealth_vars pwobani pwjbani pwsupri pwsupwi pwoccdi pwjccdi pwhecdi pwothdi pwhmvai pwhmdti pwopvai pwopdti pwcbani pwcaini pweqini pwbusvi pwbusdi pwtrusi pwinsui pwcolli pwvechi pwobdti pwtbani pwsupei pwtpvi pwccdti pwtpdi pwfini pwnfii pwassei pwdebti pw_net_worth ex_super	// creating a local macro that includes all personal wealth variables

foreach w of local wealth_vars {
	gen dw_final_bm_new_`w' = `w' * benchmarked_new_hhwte * final_deathrate	// benchmark-weighted final estate wealth
}	// multiplying each of the personal wealth variables by the enumerated person sample weights

gen bm_new_hhwte_final_deathrate = final_deathrate * benchmarked_new_hhwte	// multiplying the death rate variable by the benchmarked population-level enumerated person weights that include residents of non-private dwellings

save WAD_expected_deaths_big_final_low_value, replace

***************************************************************************************
*** Keeping only low-value estates that are likely not to have gone through probate ***
***************************************************************************************

keep if (net_property <= 0 & pwtbani <= 50000 & pweqini <= 50000 | ex_super <= 0)	// excluding those whose death is unlikely to result in a probate application

collapse(sum) `wealth_vars' dw* final_deathrate bm_new_hhwte_final_deathrate, by(year)	// collapsing (summing) the weighted and unweighted final estate wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

save HILDA_wealth_at_death_final_low_value, replace	// saving the final estates dataset

keep year dw_final_bm_new_ex_super dw_final_bm_new_pw_net_worth final_deathrate bm_new_hhwte_final_deathrate	// keeping variables that will be used for plotting

save HILDA_final_estates_low_value, replace	// saving the small final estates dataset

********************************************
*** Merging with the All Estates Dataset ***
********************************************

merge 1:1 year using HILDA_estates_low_value, nogen	// merging in the all estates dataset

************************************
*** Scaling Single Person Deaths ***
************************************

merge 1:1 year using scale_factor, nogen // matching the proportion of deaths that are single persons in the sample to the population

local final_estate_vars dw_final_bm_new_ex_super dw_final_bm_new_pw_net_worth final_deathrate bm_new_hhwte_final_deathrate	// creating a local macro with all aggregate final estate variables and the single person death rates

foreach w in `final_estate_vars' {
	replace `w' = `w' * scale_factor
}	// scaling the final estate variables by the single person death percentage scaling factor

save HILDA_all_estates_low_value, replace	// saving the combined dataset (w property)

keep year dw_final_bm_new_pw_net_worth dw_final_bm_new_ex_super bm_new_hhwte_final_deathrate dw_bm_new_hhwte_pw_net_worth dw_bm_new_hhwte_ex_super benchmarked_new_hhwte_deathrate scale_factor	// keeping relevant variables

save HILDA_lv_estates